"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.UniqueFieldDefinitionNames = exports.existedFieldDefinitionNameMessage = exports.duplicateFieldDefinitionNameMessage = void 0;
const graphql_1 = require("graphql");
const utils_1 = require("../../utils");
function duplicateFieldDefinitionNameMessage(typeName, fieldName) {
    return `Field "${typeName}.${fieldName}" can only be defined once.`;
}
exports.duplicateFieldDefinitionNameMessage = duplicateFieldDefinitionNameMessage;
function existedFieldDefinitionNameMessage(typeName, fieldName, serviceName) {
    return `${utils_1.logServiceAndType(serviceName, typeName, fieldName)}Field "${typeName}.${fieldName}" already exists in the schema. It cannot also be defined in this type extension. If this is meant to be an external field, add the \`@external\` directive.`;
}
exports.existedFieldDefinitionNameMessage = existedFieldDefinitionNameMessage;
function UniqueFieldDefinitionNames(context) {
    const schema = context.getSchema();
    const existingTypeMap = schema
        ? schema.getTypeMap()
        : Object.create(null);
    const knownFieldNames = Object.create(null);
    const possibleValueTypes = Object.create(null);
    return {
        InputObjectTypeExtension: checkFieldUniqueness,
        InterfaceTypeExtension: checkFieldUniqueness,
        ObjectTypeExtension: checkFieldUniqueness,
        InputObjectTypeDefinition: checkFieldUniquenessExcludingValueTypes,
        InterfaceTypeDefinition: checkFieldUniquenessExcludingValueTypes,
        ObjectTypeDefinition: checkFieldUniquenessExcludingValueTypes,
    };
    function checkFieldUniqueness(node) {
        const typeName = node.name.value;
        if (!knownFieldNames[typeName]) {
            knownFieldNames[typeName] = Object.create(null);
        }
        if (!node.fields) {
            return false;
        }
        const fieldNames = knownFieldNames[typeName];
        for (const fieldDef of node.fields) {
            const fieldName = fieldDef.name.value;
            if (hasField(existingTypeMap[typeName], fieldName)) {
                context.reportError(new graphql_1.GraphQLError(existedFieldDefinitionNameMessage(typeName, fieldName, existingTypeMap[typeName].astNode.serviceName), fieldDef.name));
            }
            else if (fieldNames[fieldName]) {
                context.reportError(new graphql_1.GraphQLError(duplicateFieldDefinitionNameMessage(typeName, fieldName), [fieldNames[fieldName], fieldDef.name]));
            }
            else {
                fieldNames[fieldName] = fieldDef.name;
            }
        }
        return false;
    }
    function checkFieldUniquenessExcludingValueTypes(node) {
        const typeName = node.name.value;
        const valueTypeFromSchema = existingTypeMap[typeName] &&
            existingTypeMap[typeName].astNode;
        const duplicateTypeNode = valueTypeFromSchema || possibleValueTypes[node.name.value];
        if (duplicateTypeNode) {
            const { fields } = utils_1.diffTypeNodes(node, duplicateTypeNode);
            if (Object.values(fields).every(diffEntry => diffEntry.length === 2)) {
                return false;
            }
        }
        else {
            possibleValueTypes[node.name.value] = node;
        }
        if (!knownFieldNames[typeName]) {
            knownFieldNames[typeName] = Object.create(null);
        }
        if (!node.fields) {
            return false;
        }
        const fieldNames = knownFieldNames[typeName];
        for (const fieldDef of node.fields) {
            const fieldName = fieldDef.name.value;
            if (hasField(existingTypeMap[typeName], fieldName)) {
                context.reportError(new graphql_1.GraphQLError(existedFieldDefinitionNameMessage(typeName, fieldName, existingTypeMap[typeName].astNode.serviceName), fieldDef.name));
            }
            else if (fieldNames[fieldName]) {
                context.reportError(new graphql_1.GraphQLError(duplicateFieldDefinitionNameMessage(typeName, fieldName), [fieldNames[fieldName], fieldDef.name]));
            }
            else {
                fieldNames[fieldName] = fieldDef.name;
            }
        }
        return false;
    }
}
exports.UniqueFieldDefinitionNames = UniqueFieldDefinitionNames;
function hasField(type, fieldName) {
    if (graphql_1.isObjectType(type) || graphql_1.isInterfaceType(type) || graphql_1.isInputObjectType(type)) {
        return Boolean(type.getFields()[fieldName]);
    }
    return false;
}
//# sourceMappingURL=uniqueFieldDefinitionNames.js.map