// String utilities
import { isArray, isPlainObject, isString, isUndefinedOrNull } from './inspect'; // --- Constants ---

var RX_TRIM_LEFT = /^\s+/;
var RX_TRIM_RIGHT = /\s+$/;
var RX_REGEXP_REPLACE = /[-/\\^$*+?.()|[\]{}]/g;
var RX_UN_KEBAB = /-(\w)/g;
var RX_HYPHENATE = /\B([A-Z])/g; // --- Utilities ---
// Converts PascalCase or camelCase to kebab-case

export var kebabCase = function kebabCase(str) {
  return str.replace(RX_HYPHENATE, '-$1').toLowerCase();
}; // Converts a kebab-case or camelCase string to PascalCase

export var pascalCase = function pascalCase(str) {
  str = kebabCase(str).replace(RX_UN_KEBAB, function (_, c) {
    return c ? c.toUpperCase() : '';
  });
  return str.charAt(0).toUpperCase() + str.slice(1);
}; // Lowercases the first letter of a string and returns a new string

export var lowerFirst = function lowerFirst(str) {
  str = isString(str) ? str.trim() : String(str);
  return str.charAt(0).toLowerCase() + str.slice(1);
}; // Uppercases the first letter of a string and returns a new string

export var upperFirst = function upperFirst(str) {
  str = isString(str) ? str.trim() : String(str);
  return str.charAt(0).toUpperCase() + str.slice(1);
}; // Escape characters to be used in building a regular expression

export var escapeRegExp = function escapeRegExp(str) {
  return str.replace(RX_REGEXP_REPLACE, '\\$&');
}; // Convert a value to a string that can be rendered
// `undefined`/`null` will be converted to `''`
// Plain objects and arrays will be JSON stringified

export var toString = function toString(val) {
  var spaces = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : 2;
  return isUndefinedOrNull(val) ? '' : isArray(val) || isPlainObject(val) && val.toString === Object.prototype.toString ? JSON.stringify(val, null, spaces) : String(val);
}; // Remove leading white space from a string

export var trimLeft = function trimLeft(str) {
  return toString(str).replace(RX_TRIM_LEFT, '');
}; // Remove Trailing white space from a string

export var trimRight = function trimRight(str) {
  return toString(str).replace(RX_TRIM_RIGHT, '');
}; // Remove leading and trailing white space from a string

export var trim = function trim(str) {
  return toString(str).trim();
}; // Lower case a string

export var lowerCase = function lowerCase(str) {
  return toString(str).toLowerCase();
}; // Upper case a string

export var upperCase = function upperCase(str) {
  return toString(str).toUpperCase();
};